"use strict";
//
// Copyright 2023 Signal Messenger, LLC.
// SPDX-License-Identifier: AGPL-3.0-only
//
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.sanitize = exports.SanitizedMetadata = void 0;
/**
 * An MP4 format “sanitizer”.
 *
 * Currently the sanitizer always performs the following functions:
 *
 * - Return all presentation metadata present in the input as a self-contained contiguous byte array.
 * - Find and return a pointer to the span in the input containing the (contiguous) media data.
 *
 * “Presentation” metadata means any metadata which is required by an MP4 player to play the file. “Self-contained and
 * contiguous” means that the returned metadata can be concatenated with the media data to form a valid MP4 file.
 *
 * The original metadata may or may not need to be modified in order to perform these functions. In the case that the
 * original metadata does not need to be modified, the returned sanitized metadata will be null to prevent needless data
 * copying.
 *
 * ## Unsupported MP4 features
 *
 * The sanitizer does not currently support:
 *
 * - “Fragmented” MP4 files, which are mostly used for adaptive-bitrate streaming.
 * - Discontiguous media data, i.e. media data (mdat) boxes interspersed with presentation metadata (moov).
 * - Media data references (dref) pointing to separate files.
 * - Any similar format, e.g. Quicktime File Format (mov) or the legacy MP4 version 1, which does not contain the "isom"
 *   compatible brand in its file type header (ftyp).
 *
 * @module Mp4Sanitizer
 */
const Native = require("../Native");
const BigIntUtil_1 = require("./zkgroup/internal/BigIntUtil");
class SanitizedMetadata {
    constructor(handle) {
        this._nativeHandle = handle;
    }
    static _fromNativeHandle(handle) {
        return new SanitizedMetadata(handle);
    }
    /**
     * Get the sanitized metadata, if any.
     * @return The sanitized metadata, or {@code null} if it didn't need to be sanitized.
     */
    getMetadata() {
        const metadata = Native.SanitizedMetadata_GetMetadata(this);
        if (metadata.length == 0) {
            return null;
        }
        return metadata;
    }
    /**
     * Get the offset of the media data in the processed input.
     * @return The offset of the media data in the processed input.
     */
    getDataOffset() {
        const buffer = Native.SanitizedMetadata_GetDataOffset(this);
        return buffer.readBigUInt64BE();
    }
    /**
     * Get the length of the media data in the processed input.
     * @return The length of the media data in the processed input.
     */
    getDataLen() {
        const buffer = Native.SanitizedMetadata_GetDataLen(this);
        return buffer.readBigUInt64BE();
    }
}
exports.SanitizedMetadata = SanitizedMetadata;
/**
 * Sanitize an MP4 input.
 *
 * @param input An MP4 format input stream.
 * @param length The exact length of the input stream.
 * @return The sanitized metadata.
 * @throws IoError If an IO error on the input occurs.
 * @throws InvalidMediaInputError If the input could not be parsed because it was invalid.
 * @throws UnsupportedMediaInputError If the input could not be parsed because it's unsupported in some way.
 */
function sanitize(input, len) {
    return __awaiter(this, void 0, void 0, function* () {
        const sanitizedMetadataNativeHandle = yield Native.Mp4Sanitizer_Sanitize(input, (0, BigIntUtil_1.bufferFromBigUInt64BE)(len));
        return SanitizedMetadata._fromNativeHandle(sanitizedMetadataNativeHandle);
    });
}
exports.sanitize = sanitize;
//# sourceMappingURL=Mp4Sanitizer.js.map